<?php
/**
 * @package rsvp-pro
 * @subpackage includes/admin
 * @author Swim or Die Software
 * @since 3.5.2
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) :
	exit;
endif;

// Load WP_List_Table if not loaded
if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * rsvp_pro_message_log_table Class
 *
 * Renders the Message Log table on the Message Log page
 *
 * @since 3.2.5
 */
class Rsvp_Pro_Message_Log_Table extends WP_List_Table {
	/**
	 * Number of items per page
	 *
	 * @var int
	 */
	public $per_page = 100;

	/**
	 * Number of message log entries found
	 *
	 * @var int
	 */
	public $count = 0;

	/**
	 * Total message log entries
	 *
	 * @var int
	 */
	public $total = 0;

	/**
	 * The event ID we are getting the list for
	 *
	 * @var int
	 */
	private $event_id;

	/**
	 * Get things started
	 *
	 * @since 3.5.2
	 * @see WP_List_Table::__construct()
	 */
	public function __construct( $event_id ) {
		global $status, $page;

		$this->event_id = $event_id;

		// Set parent defaults
		parent::__construct(
			array(
				'singular' => rsvp_pro_get_admin_string( 'message_log_table_label' ),
				'plural'   => rsvp_pro_get_admin_string( 'message_logs_table_label' ),
				'ajax'     => false,
			)
		);

	}

	/**
	 * This function renders most of the columns in the list table.
	 *
	 * @access public
	 * @since 3.2.5
	 *
	 * @param array  $reminder Contains all the data of the reminder
	 * @param string $column_name The name of the column
	 *
	 * @return string Column Name
	 */
	function column_default( $reminder, $column_name ) {
		return esc_html( $reminder[ $column_name ] );
	}

	/**
	 * Renders the column for the subject column so we can add in a link to edit
	 *
	 * @param  array $message_log Contains all the data of the message log
	 * @return string Column data
	 */
	function column_subject( $message_log ) {
		return '<a href="' . esc_url(
			remove_query_arg(
				array( '_wpnonce', '_wp_http_referer', 'message_log' ),
				add_query_arg(
					array(
						'action' => 'view_message_log',
						'log_id' => $message_log['id'],
					)
				)
			)
		) . '">' . esc_html( $message_log['subject'] ) . '</a>';
	}

	/**
	 * Renders the send date column with a date format that matches the WP install specified format
	 *
	 * @param  array $message_log Contains all the data of the message log
	 * @return string Column The properly formatted date
	 */
	function column_send_date( $message_log ) {
		return date_i18n( get_option( 'date_format' ), strtotime( $message_log['send_date'] ) );
	}

	/**
	 * Renders the action link for an individual message log entry
	 *
	 * @param  array $message_log Contains all the data of the message log
	 * @return string Action Column The action link for a specific message log
	 */
	function column_action( $message_log ) {
		return '<a href="' . esc_url(
			remove_query_arg(
				array( '_wpnonce', '_wp_http_referer', 'message_log' ),
				add_query_arg(
					array(
						'action'         => 'mass_email',
						'message_log_id' => $message_log['id'],
					)
				)
			)
		) . '">' . rsvp_pro_get_admin_string( 'message_log_resend_label' ) . '</a>';
	}

	/**
	 * Gets the name of the primary column.
	 *
	 * @since 2.2.4
	 * @access protected
	 *
	 * @return string Name of the primary column.
	 */
	protected function get_primary_column_name() {
		return 'subject';
	}

	/**
	 * Render the checkbox column
	 *
	 * @access public
	 * @since 3.5.2
	 * @param array $message_log Contains all the data for the checkbox column
	 * @return string Displays a checkbox
	 */
	function column_cb( $message_log ) {
		return sprintf(
			'<input type="checkbox" name="%1$s[]" value="%2$s" />',
			/*$1%s*/ 'message_log',
			/*$2%s*/ $message_log['id']
		);
	}

	/**
	 * Retrieve the table columns
	 *
	 * @access public
	 * @since 3.5.2
	 * @return array $columns Array of all the list table columns
	 */
	public function get_columns() {
		$columns = array(
			'cb'        => '<input type="checkbox" />',
			'subject'   => rsvp_pro_get_admin_string( 'message_subject_label' ),
			'send_date' => rsvp_pro_get_admin_string( 'message_log_sent_on_label' ),
			'action'    => rsvp_pro_get_admin_string( 'message_log_copy_table_label' ),
		);

		return $columns;

	}

	/**
	 * Outputs the bulk actions for the message log entries
	 *
	 * @access public
	 * @since 3.5.2
	 * @return array $actions Array of the bulk actions
	 */
	public function get_bulk_actions( $which = '' ) {
		$actions = array(
			'delete_message_logs' => rsvp_pro_get_admin_string( 'delete_label' ),
		);

		return $actions;
	}

	/**
	 * Retrieves the current page number
	 *
	 * @access public
	 * @since 3.5.2
	 * @return int Current page number
	 */
	public function get_page_number() {
		return isset( $_GET['paged'] ) ? absint( $_GET['paged'] ) : 1;
	}

	/**
	 * Retrieves the search query string
	 *
	 * @access public
	 * @since 3.5.2
	 * @return mixed string If search is present, false otherwise
	 */
	public function get_search() {
		// TODO: IMplement
		return ! empty( $_GET['s'] ) ? urldecode( trim( $_GET['s'] ) ) : false;
	}

	/**
	 * Setup the final data for the table
	 *
	 * @access public
	 * @since 3.5.2
	 * @uses RSVP_pro_Message_Log_Table::get_columns()
	 * @uses WP_List_Table::get_sortable_columns()
	 * @uses RSVP_pro_Message_Log_Table::get_page_number()
	 * @uses RSVP_pro_Message_Log::get_data_for_list()
	 * @uses RSVP_pro_Message_Log::get_total_count()
	 * @return void
	 */
	public function prepare_items() {

		$columns  = $this->get_columns();
		$hidden   = array(); // No hidden columns
		$sortable = $this->get_sortable_columns();

		$this->_column_headers = array( $columns, $hidden, $sortable );

		$message_logs = new Rsvp_Pro_Message_Logs( $this->event_id );

		$this->items = $message_logs->get_data_for_list( $this->get_page_number() );
		$this->total = $message_logs->get_total_count();

		$this->set_pagination_args(
			array(
				'total_items' => $this->total,
				'per_page'    => $this->per_page,
				'total_pages' => ceil( $this->total / $this->per_page ),
			)
		);
	}

}
