<?php
/**
 * @package rsvp-pro
 * @subpackage includes/admin
 * @author Swim or Die Software
 * @since 3.5.2
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) :
	exit;
endif;

/**
 * Class for handling message logs
 *
 * @since 3.5.2
 */
class Rsvp_Pro_Message_Logs {


	/**
	 * The event ID we want to get the message logs for
	 *
	 * @var int
	 */
	private $event_id;


	public function __construct( $event_id ) {
		if ( ! is_numeric( $event_id ) || ( $event_id <= 0 ) ) {
			die( '<b>Please specify an event ID</b>' );
		}

		$this->event_id = $event_id;
		$this->load_dependecies();
	}

	/**
	 * Loads the other files and classes that will be used by this class
	 *
	 * @since 3.5.2
	 */
	private function load_dependecies() {
		require_once 'class_rsvp_pro_message_log_table.php';
	}

	/**
	 * Display function for the message log list
	 *
	 * @since 3.5.2
	 */
	public function list_message_logs() {
		$message_logs_table = new Rsvp_Pro_Message_Log_Table( $this->event_id );
		$message_logs_table->prepare_items();
		?>
		<div class="wrap">
			<h1>
			<?php
			echo rsvp_pro_get_admin_string( 'message_logs_table_label' ) . ' ';
			echo rsvp_pro_get_event_name( $this->event_id )
			?>
			</h1>
			<form id="rsvp-pro-message-logs-filter" method="get" action="<?php echo admin_url( 'admin.php?page=rsvp-pro-top-level&action=message_logs&eventID=' . $this->event_id ); ?>">
			<input type="hidden" name="page" value="rsvp-pro-top-level" />
			<input type="hidden" name="action" value="message_logs" />
			<input type="hidden" name="eventID" value="<?php echo esc_attr( $this->event_id ); ?>" />
			<?php $message_logs_table->views(); ?>
			<?php $message_logs_table->display(); ?>
		  </form>
		</div>
		<?php
	}

	public function view_message_log( $message_log_id ) {
		global $wpdb;

		$sql         = 'SELECT id, from_email, from_name, rsvp_status, read_receipt, 
                subject, message, send_date 
                FROM ' . PRO_MESSAGE_LOG . ' 
                WHERE id = %d';
		$message_log = $wpdb->get_row( $wpdb->prepare( $sql, $message_log_id ) );
		?>
		<div class="wrap">
			<h2><?php echo rsvp_pro_get_admin_string( 'message_log_table_label' ); ?> - 
				<?php echo esc_html( stripslashes( $message_log->subject ) ); ?><a href="
						   <?php
							echo esc_url(
								remove_query_arg(
									array( '_wpnonce', '_wp_http_referer', 'message_log' ),
									add_query_arg(
										array(
											'action' => 'mass_email',
											'message_log_id' => $message_log_id,
										)
									)
								)
							);
							?>
																							" class="add-new-h2"><?php echo rsvp_pro_get_admin_string( 'message_log_resend_label' ); ?></a></h2>
			<table class="form-table">
				<tbody>
					<tr>
						<th scope="row" valign="top">
							<label><?php echo rsvp_pro_get_admin_string( 'message_log_sent_on_label' ); ?></label>
						</th>
						<td>
							<?php echo date_i18n( get_option( 'date_format' ), strtotime( $message_log->send_date ) ); ?>
						</td>
					</tr>
					<tr>
						<th scope="row" valign="top">
							<label><?php echo rsvp_pro_get_admin_string( 'from_email_label' ); ?></label>
						</th>
						<td>
							<?php echo esc_html( stripslashes( $message_log->from_email ) ); ?>
						</td>
					</tr>
					<tr>
						<th scope="row" valign="top">
							<label><?php echo rsvp_pro_get_admin_string( 'from_email_name_label' ); ?></label>
						</th>
						<td>
							<?php echo esc_html( stripslashes( $message_log->from_name ) ); ?>
						</td>
					</tr>
					<tr>
						<th scope="row" valign="top">
							<label><?php echo rsvp_pro_get_admin_string( 'limit_rsvp_status_label' ); ?></label>
						</th>
						<td>
							<?php echo rsvp_pro_humanize_rsvp_status( $message_log->rsvp_status ); ?>
						</td>
					</tr>
					<tr>
						<th scope="row" valign="top">
							<label><?php echo rsvp_pro_get_admin_string( 'read_receipt_label' ); ?></label>
						</th>
						<td>
							<?php echo ( $message_log->read_receipt == 'Y' ) ? 'Yes' : 'No'; ?>
						</td>
					</tr>
					<tr>
						<th scope="row" valign="top">
							<label><?php echo rsvp_pro_get_admin_string( 'message_subject_label' ); ?></label>
						</th>
						<td>
							<?php echo esc_html( stripslashes( $message_log->subject ) ); ?>
						</td>
					</tr>
					<tr>
						<th scope="row" valign="top">
							<label><?php echo rsvp_pro_get_admin_string( 'message_email_label' ); ?></label>
						</th>
						<td>
							<?php echo esc_html( stripslashes( $message_log->message ) ); ?>
						</td>
					</tr>
					<tr>
						<th scope="row" valign="top">
							<label><?php echo rsvp_pro_get_admin_string( 'message_attendees_emailed_label' ); ?></label>
						</th>
						<td>
							<?php
							$sql       = 'SELECT firstName, lastName 
                                FROM ' . PRO_ATTENDEES_TABLE . ' a 
                                INNER JOIN ' . PRO_MESSAGE_LOG_ATTENDEES . ' mla ON mla.attendeeID = a.id 
                                WHERE mla.message_id = %d';
							$attendees = $wpdb->get_results( $wpdb->prepare( $sql, $message_log_id ) );
							foreach ( $attendees as $a ) {
								?>
								<p><?php echo esc_html( stripslashes( $a->firstName . ' ' . $a->lastName ) ); ?></p>
								<?php
							}
							?>
						</td>
					</tr>
				</tbody>
			</table>
		</div>
		<?php
	}

	/**
	 * Gets the data for the admin table
	 *
	 * @since 3.5.2
	 * @param int $page The current page we are getting data for
	 * @return array $data An array with the data
	 */
	public function get_data_for_list( $page = 1 ) {
		global $wpdb;

		$offset = ( $page - 1 ) * 100;

		$sql  = 'SELECT id, rsvpEventID, from_email, from_name, rsvp_status, 
                    read_receipt, subject, message, send_date
                FROM  ' . PRO_MESSAGE_LOG . '
                WHERE rsvpEventID = %d 
                LIMIT ' . $offset . ', 100 ';
		$data = $wpdb->get_results( $wpdb->prepare( $sql, $this->event_id ), ARRAY_A );

		return $data;
	}

	/**
	 * Gets the total count for the message logs
	 *
	 * @since 3.5.2
	 * @return int the total number of message logs for this event
	 */
	public function get_total_count() {
		global $wpdb;

		$sql   = 'SELECT COUNT(id) 
                FROM ' . PRO_MESSAGE_LOG . ' 
                WHERE rsvpEventID = %d';
		$count = $wpdb->get_var( $wpdb->prepare( $sql, $this->event_id ) );

		return $count;
	}
}
