<?php
/**
 * @package rsvp-pro
 * @subpackage includes/admin
 * @author WPChill
 * @since 2.2.4
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) :
	exit;
endif;

/**
 * Class for handling reminders
 *
 * @since 2.2.4
 */
class Rsvp_Pro_Event_Reminders {


	/**
	 * The event ID we want to get the reminders for
	 *
	 * @var int
	 */
	private $eventID;


	public function __construct( $eventID ) {
		if ( ! is_numeric( $eventID ) || ( $eventID <= 0 ) ) {
			die( '<b>Please specify an event ID</b>' );
		}

		$this->eventID = $eventID;
		$this->load_dependecies();
	}

	/**
	 * Loads the other files and classes that will be used by this class
	 *
	 * @since 2.2.4
	 */
	private function load_dependecies() {
		require_once 'class_rsvp_pro_reminders_table.php';
	}

	/**
	 * Display function for the reminder list
	 *
	 * @since 2.2.4
	 */
	public function list_reminders() {
		$reminders_table = new Rsvp_Pro_Reminders_Table( $this->eventID );
		$reminders_table->prepare_items(); ?>
	<div class="wrap">
		<h1>
		<?php
		echo rsvp_pro_get_admin_string( 'reminders_for_label' ) . ' ';
		echo rsvp_pro_get_event_name( $this->eventID )
		?>
		<a href="<?php echo esc_url( add_query_arg( array( 'action' => 'reminder-form' ) ) ); ?>" class="add-new-h2"><?php echo rsvp_pro_get_admin_string( 'add_new_label' ); ?></a></h1>
		<p class="description">
		<?php
		echo rsvp_pro_get_admin_string( 'the_date_label' ) . ' ';
		echo '<b>' . $this->get_reminder_base_date() . '</b> ';
		echo rsvp_pro_get_admin_string( 'used_for_reminder_calculation_label' ) . ' ' .
		rsvp_pro_get_admin_string( 'date_retrieved_from_label' );
		echo '<b>' . $this->get_reminder_date_field_used() . '</b> ';
		echo rsvp_pro_get_admin_string( 'field_label' );
		?>
		</p>
		<form id="rsvp-pro-reminders-filter" method="get" action="<?php echo admin_url( 'admin.php?page=rsvp-pro-top-level&action=reminders&eventID=' . $this->eventID ); ?>">
		<input type="hidden" name="page" value="rsvp-pro-top-level" />
		<input type="hidden" name="action" value="reminders" />
		<input type="hidden" name="eventID" value="<?php echo esc_attr( $this->eventID ); ?>" />

		<?php $reminders_table->views(); ?>
		<?php $reminders_table->display(); ?>
	  </form>
	</div>
		<?php
	}

	/**
	 * Grabs the reminder base date that will be used to calculate when reminders will be sent.
	 *
	 * @return the date
	 */
	public function get_reminder_base_date() {
		$baseDate   = rsvp_pro_get_event_information( $this->eventID, RSVP_PRO_INFO_CLOSE_DATE );
		$event_date = rsvp_pro_get_event_information( $this->eventID, RSVP_PRO_INFO_EVENT_START_DATE );

		if ( strtotime( $event_date ) ) {
			$baseDate = $event_date;
		}

		return date_i18n( get_option( 'date_format' ), strtotime( $baseDate ) );
	}

	public function get_reminder_date_field_used() {
		if ( strtotime( rsvp_pro_get_event_information( $this->eventID, RSVP_PRO_INFO_EVENT_START_DATE ) ) ) {
			return rsvp_pro_get_admin_string( 'settings_event_start_date_label' );
		} else {
			return rsvp_pro_get_admin_string( 'settings_rsvp_close_date_label' );
		}
	}


	public function show_form( $reminderID ) {
		$action     = rsvp_pro_get_admin_string( 'add_reminder_label' );
		$reminderID = absint( $reminderID );

		if ( ! class_exists( 'Rsvp_Pro_Event_Reminder_Dao' ) ) {
			require_once 'class_rsvp_pro_reminder_dao.php';
		}

		if ( is_numeric( $reminderID ) && ( $reminderID > 0 ) ) {
			$action = rsvp_pro_get_admin_string( 'edit_reminder_label' );
		}

		$reminder = new Rsvp_Pro_Event_Reminder_Dao( $reminderID );
		?>
		<h2><?php echo $action; ?> - <a href="<?php echo admin_url( 'admin.php?page=rsvp-pro-top-level&action=reminders&eventID=' . $this->eventID ); ?>" class="button-secondary"><?php echo rsvp_pro_get_admin_string( 'go_back_label' ); ?></a></h2>
		<form id="rsvp-pro-reminder-form" action="" method="post">
			<table class="form-table">
				<tbody>
					<tr>
						<th scope="row" valign="top">
							<label for="reminder-name"><?php echo rsvp_pro_get_admin_string( 'reminder_name_label' ); ?></label>
						</th>
						<td>
							<input name="name" id="reminder-name" type="text" value="<?php echo esc_attr( $reminder->get_Name() ); ?>" style="width: 300px;" required/>
							<p class="description"><?php echo rsvp_pro_get_admin_string( 'reminder_name_description' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row" valign="top">
							<label for="reminder-time-type"><?php echo rsvp_pro_get_admin_string( 'reminder_time_label' ); ?></label>
						</th>
						<td>
							<?php echo rsvp_pro_get_admin_string( 'send_reminder_label' ) . ' '; ?>
							<input type="text" name="period" id="reminder-period" value="<?php echo esc_attr( $reminder->get_Period() ); ?>" placeholder="e.g. 2" />
							<select name="timeType" id="reminder-time-type" size="1">
								<option value="">--</option>
								<option value="day"
								<?php echo ( $reminder->get_TimeType() == 'day' ) ? 'selected="selected"' : ''; ?>><?php echo rsvp_pro_get_admin_string( 'settings_day_label' ); ?></option>
								<option value="week"
								<?php echo ( $reminder->get_TimeType() == 'week' ) ? 'selected="selected"' : ''; ?>><?php echo rsvp_pro_get_admin_string( 'settings_week_label' ); ?></option>
								<option value="month"
								<?php echo ( $reminder->get_TimeType() == 'month' ) ? 'selected="selected"' : ''; ?>><?php echo rsvp_pro_get_admin_string( 'settings_month_label' ); ?></option>
							</select>
							<?php
							echo rsvp_pro_get_admin_string( 'before_label' ) . ' ' .
									$this->get_reminder_base_date();
							?>
						</td>
					</tr>
					<tr>
						<th scope="row" valign="top">
							<label for="reminder-sendon"><?php echo rsvp_pro_get_admin_string( 'send_reminder_on_label' ); ?></label>
						</th>
						<td>
							<input type="text" name="sendon" id="reminder-sendon" value="<?php echo esc_attr( $reminder->get_SendOn() ); ?>" />
						</td>
					</tr>
					<tr>
						<th scope="row" valign="top">
							<label for="reminder-rsvp-statuses"><?php echo rsvp_pro_get_admin_string( 'reminders_only_rsvp_status_label' ); ?></label>
						</th>
						<td>
							<span><?php echo rsvp_pro_get_admin_string( 'possible_statuses_to_send_label' ); ?></span>
							<span style="margin-left:32px;"><?php echo rsvp_pro_get_admin_string( 'reminders_sent_to_statuses_label' ); ?></span>
							<select name="rsvpStatuses[]" id="reminder-rsvp-statuses" multiple="multiple" size="5">
								<?php
									$statuses         = array( 'Yes', 'No', 'NoResponse', 'Waitlist', 'Maybe' );
									$selectedStatuses = explode( ',', $reminder->get_RsvpStatuses() );
								foreach ( $statuses as $s ) {
									?>
										<option value="<?php echo $s; ?>" <?php echo ( in_array( $s, $selectedStatuses ) ) ? 'selected="selected"' : ''; ?>><?php echo $s; ?></option>
								<?php } ?>
							</select>
						</td>
					</tr>
					<?php
						rsvp_pro_admin_message_variables( $this->eventID );
					?>
					<tr>
						<th scope="row" valign="top">
							<label for="reminder-message-subject"><?php echo rsvp_pro_get_admin_string( 'reminder_subject_label' ); ?></label>
						</th>
						<td>
							<input type="text" name="messageSubject" id="reminder-message-subject" value="<?php echo esc_attr( $reminder->get_MessageSubject() ); ?>" class="large-text" />
						</td>
					</tr>
					<tr>
						<th scope="row" valign="top">
							<label for="messageBody"><?php echo rsvp_pro_get_admin_string( 'reminder_message_label' ); ?></label>
						</th>
						<td>
							<?php wp_editor( $reminder->get_MessageBody(), 'messageBody', $settings = array() ); ?>
						</td>
					</tr>
				</tbody>
			</table>
			<p class="submit">
				<input type="hidden" name="page" value="rsvp-pro-top-level" />
				<input type="hidden" name="action" value="save_reminder"/>
				<input type="hidden" name="reminderID" value="<?php echo absint( $reminderID ); ?>"/>
				<input type="hidden" name="rsvp-reminder-nonce" value="<?php echo wp_create_nonce( 'rsvp-reminder-form' ); ?>"/>
				<input type="hidden" name="eventID" value="<?php echo absint( $this->eventID ); ?>" />
				<input type="submit" value="<?php echo rsvp_pro_get_admin_string( 'save_reminder_label' ); ?>" class="button-primary"/>
			</p>
		</form>
		<?php
	}

	/**
	 * Gets the data for the admin table
	 *
	 * @since 2.2.4
	 * @return array $data An array with the data
	 */
	public function get_data_for_list() {
		global $wpdb;

		$sql  = 'SELECT id, rsvpEventID, name, timeType, sendOn, period, rsvpStatuses 
                FROM ' . PRO_EVENT_REMINDERS_TABLE . ' 
                WHERE rsvpEventID = %d';
		$data = $wpdb->get_results( $wpdb->prepare( $sql, $this->eventID ), ARRAY_A );

		return $data;
	}

	/**
	 * Gets the total count for the reminders
	 *
	 * @since 2.2.4
	 * @return int the total number of reminders for this event
	 */
	public function get_total_count() {
		global $wpdb;

		$sql   = 'SELECT COUNT(id) 
                FROM ' . PRO_EVENT_REMINDERS_TABLE . ' 
                WHERE rsvpEventID = %d';
		$count = $wpdb->get_var( $wpdb->prepare( $sql, $this->eventID ) );

		return $count;
	}
}
