<?php
/**
 * Class file that is wrapper for attendee custom question answers
 *
 * @since 3.8.8
 * @package rsvp-pro-plugin
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) :
	exit;
endif;

/**
 * Class for handling attendee custom question answers and storing it into the database.
 *
 * @since 3.8.8
 */
class Rsvp_Pro_Attendee_Answers {

	/**
	 * The ID for this recorded answer.
	 *
	 * @var integer
	 */
	private $id = 0;

	/**
	 * The answer the attendee gave.
	 *
	 * @var string
	 */
	private $answer = '';

	/**
	 * The custom question associated with this answer.
	 *
	 * @var integer
	 */
	private $custom_question_id = 0;

	/**
	 * The attendee associated with this answer.
	 *
	 * @var integer
	 */
	private $attendee_id = 0;

	/**
	 * Constructor for this answer.
	 *
	 * @param integer $id The answer we wish to load.
	 */
	public function __construct( $id = 0 ) {
		$this->id = $id;
		$this->load();
	}

	/**
	 * Loads the information related to this attendee's answer.
	 */
	private function load() {
		global $wpdb;

		$sql    = 'SELECT * FROM ' . PRO_ATTENDEE_ANSWERS . ' WHERE id = %d';
		$answer = $wpdb->get_row( $wpdb->prepare( $sql, $this->id ) );
		if ( false !== $answer ) {
			$this->attendee_id        = $answer->attendeeID;
			$this->answer             = stripslashes_deep( $answer->answer );
			$this->custom_question_id = $answer->questionID;
		}
	}

	/**
	 * Deletes the attendee answer.
	 */
	public function delete() {
		global $wpdb;
		if ( 0 < $this->id ) {
			$wpdb->delete(
				PRO_ATTENDEE_ANSWERS,
				array( 'id' => $this->id ),
				array( '%d' )
			);
		}
	}

	/**
	 * Save the information to the database.
	 */
	public function save() {
		global $wpdb;

		if ( 0 < $this->id ) {
			$wpdb->update(
				PRO_ATTENDEE_ANSWERS,
				array(
					'attendeeID' => $this->attendee_id,
					'questionID' => $this->custom_question_id,
					'answer'     => $this->answer,
				),
				array( 'id' => $this->id ),
				array(
					'%d',
					'%d',
					'%s',
				),
				array( '%d' )
			);
		} else {
			$wpdb->insert(
				PRO_ATTENDEE_ANSWERS,
				array(
					'attendeeID' => $this->attendee_id,
					'questionID' => $this->custom_question_id,
					'answer'     => $this->answer,
				),
				array(
					'%d',
					'%d',
					'%s',
				)
			);
		}
	}

	/**
	 * Creates an array of all the custom question answers for an attendee.
	 *
	 * @param  int $attendee_id The id of the attendee we want the answers for.
	 * @param  int $event_id The id of the event that we want to load answers for.
	 *
	 * @return array            An array of Rsvp_Pro_Attendee_Answers.
	 */
	public static function load_answers_for_attendee( $attendee_id, $event_id ) {
		global $wpdb;

		$answers = array();
		$sql     = 'SELECT a.id FROM ' . PRO_ATTENDEE_ANSWERS . ' a 
			JOIN ' . PRO_QUESTIONS_TABLE . ' q ON q.id = a.questionID 
			WHERE a.attendeeID = %d AND q.rsvpEventID = %d';
		$ans     = $wpdb->get_results( $wpdb->prepare( $sql, $attendee_id, $event_id ) );
		foreach ( $ans as $a ) {
			$answers[] = new Rsvp_Pro_Attendee_Answers( $a->id );
		}

		return $answers;
	}

	/**
	 * Returns the ID for this record.
	 *
	 * @return int The identifier for this record.
	 */
	public function get_id() {
		return $this->id;
	}

	/**
	 * The custom question ID that is associated with this answer.
	 *
	 * @return int The custom question identifier for this answer.
	 */
	public function get_custom_question_id() {
		return $this->custom_question_id;
	}

	/**
	 * Sets the custom question ID.
	 *
	 * @param int $question_id Sets the custom question ID for this attendee answer.
	 */
	public function set_custom_question_id( $question_id ) {
		if ( is_numeric( $question_id ) && ( 0 <= $question_id ) ) {
			$this->custom_question_id = $question_id;
		} else {
			$this->custom_question_id = 0;
		}
	}

	/**
	 * Returns the attendee ID associated with the answer.
	 *
	 * @return integer The attendee ID.
	 */
	public function get_attendee_id() {
		return $this->attendee_id;
	}

	/**
	 * Sets the attendee that should be associated with this custom question answer.
	 *
	 * @param integer $attendee_id The attendee ID we want associated with this custom question.
	 */
	public function set_attendee_id( $attendee_id ) {
		if ( is_numeric( $attendee_id ) && ( 0 <= $attendee_id ) ) {
			$this->attendee_id = $attendee_id;
		} else {
			$this->attendee_id = 0;
		}
	}

	/**
	 * Gets the answer the attendee has submitted for this custom question.
	 *
	 * @return string The answer the attendee has provided.
	 */
	public function get_answer() {
		return $this->answer;
	}

	/**
	 * Records the answer the attendee has provided.
	 *
	 * @param string $answer The answer that the user has submitted for this custom question.
	 */
	public function set_answer( $answer ) {
		$this->answer = $answer;
	}
}
