<?php
/**
 * @package rsvp-pro
 * @subpackage includes/admin
 * @author WPChill
 * @since 2.2.4
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) :
	exit;
endif;

// Load WP_List_Table if not loaded
if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * rsvp_pro_reminders_table Class
 *
 * Renders the Reminders table on the Reminders page
 *
 * @since 2.2.4
 */
class Rsvp_Pro_Reminders_Table extends WP_List_Table {
	/**
	 * Number of items per page
	 *
	 * @var int
	 */
	public $per_page = 300;

	/**
	 * Number of reminders found
	 *
	 * @var int
	 */
	public $count = 0;

	/**
	 * Total reminders
	 *
	 * @var int
	 */
	public $total = 0;

	/**
	 * The event ID we are getting the list for
	 *
	 * @var int
	 */
	private $eventID;

	/**
	 * Get things started
	 *
	 * @since 2.2.4
	 * @see WP_List_Table::__construct()
	 */
	public function __construct( $eventID ) {
		global $status, $page;

		$this->eventID = $eventID;

		// Set parent defaults
		parent::__construct(
			array(
				'singular' => rsvp_pro_get_admin_string( 'reminder_label' ),
				'plural'   => rsvp_pro_get_admin_string( 'reminders_label' ),
				'ajax'     => false,
			)
		);

	}

	/**
	 * This function renders most of the columns in the list table.
	 *
	 * @access public
	 * @since 2.2.4
	 *
	 * @param array  $reminder Contains all the data of the reminder
	 * @param string $column_name The name of the column
	 *
	 * @return string Column Name
	 */
	function column_default( $reminder, $column_name ) {
		return esc_html( $reminder[ $column_name ] );
	}

	/**
	 * Renders the column for the name column so we can add in a link to edit
	 *
	 * @param  array $reminder Contains all the data of the reminder
	 * @return string Column data
	 */
	function column_name( $reminder ) {
		return '<a href="' . esc_url(
			add_query_arg(
				array(
					'action'     => 'reminder-form',
					'reminderID' => $reminder['id'],
				)
			)
		) . '">' . esc_html( $reminder['name'] ) . '</a>';
	}

	/**
	 * Gets the name of the primary column.
	 *
	 * @since 2.2.4
	 * @access protected
	 *
	 * @return string Name of the primary column.
	 */
	protected function get_primary_column_name() {
		return 'name';
	}

	/**
	 * Render the checkbox column
	 *
	 * @access public
	 * @since 2.2.4
	 * @param array $reminder Contains all the data for the checkbox column
	 * @return string Displays a checkbox
	 */
	function column_cb( $reminder ) {
		return sprintf(
			'<input type="checkbox" name="%1$s[]" value="%2$s" />',
			/*$1%s*/ 'rsvp_reminder',
			/*$2%s*/ $reminder['id']
		);
	}

	/**
	 * Retrieve the table columns
	 *
	 * @access public
	 * @since 2.2.4
	 * @return array $columns Array of all the list table columns
	 */
	public function get_columns() {
		$columns = array(
			'cb'           => '<input type="checkbox" />',
			'name'         => rsvp_pro_get_admin_string( 'reminder_name_label' ),
			'timeType'     => rsvp_pro_get_admin_string( 'time_qualifier_label' ),
			'period'       => rsvp_pro_get_admin_string( 'time_period_label' ),
			'sendOn'       => rsvp_pro_get_admin_string( 'send_reminder_specific_date_label' ),
			'rsvpStatuses' => rsvp_pro_get_admin_string( 'relevant_rsvp_label' ),
		);

		return $columns;

	}

	/**
	 * Outputs the bulk actions for the contest codes
	 *
	 * @access public
	 * @since 1.0
	 * @return array $actions Array of the bulk actions
	 */
	public function get_bulk_actions( $which = '' ) {
		$actions = array(
			'delete_reminders' => rsvp_pro_get_admin_string( 'delete_label' ),
		);

		return $actions;
	}

	/**
	 * Retrieves the current page number
	 *
	 * @access public
	 * @since 2.2.4
	 * @return int Current page number
	 */
	public function get_page_number() {
		return isset( $_GET['paged'] ) ? absint( $_GET['paged'] ) : 1;
	}

	/**
	 * Retrieves the search query string
	 *
	 * @access public
	 * @since 1.7
	 * @return mixed string If search is present, false otherwise
	 */
	public function get_search() {
		// TODO: IMplement
		return ! empty( $_GET['s'] ) ? urldecode( trim( $_GET['s'] ) ) : false;
	}

	/**
	 * Setup the final data for the table
	 *
	 * @access public
	 * @since 2.2.4
	 * @uses Rsvp_Pro_Reminders_Table::get_columns()
	 * @uses WP_List_Table::get_sortable_columns()
	 * @uses Rsvp_Pro_Reminders_Table::get_page_number()
	 * @uses Rsvp_Pro_Event_Reminders::get_data_for_list()
	 * @uses Rsvp_Pro_Event_Reminders::get_total_count()
	 * @return void
	 */
	public function prepare_items() {

		$columns  = $this->get_columns();
		$hidden   = array(); // No hidden columns
		$sortable = $this->get_sortable_columns();

		$this->_column_headers = array( $columns, $hidden, $sortable );

		$reminders = new Rsvp_Pro_Event_Reminders( $this->eventID );

		// NOTE: This doesn't take into account for pagination, I am guessing people will not need that many reminders...
		$this->items = $reminders->get_data_for_list();
		$this->total = $reminders->get_total_count();

		$this->set_pagination_args(
			array(
				'total_items' => $this->total,
				'per_page'    => $this->per_page,
				'total_pages' => ceil( $this->total / $this->per_page ),
			)
		);
	}

}
