<?php
/**
 * Class file that is wrapper for attendee change log entries
 *
 * @since 3.8.8
 * @package rsvp-pro-plugin
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) :
	exit;
endif;

/**
 * Class for handling attendee change log entries and storing it into the database.
 *
 * @since 3.8.8
 */
class Rsvp_Pro_Attendee_Changelog {

	/**
	 * The identifier for this record.
	 *
	 * @var integer
	 */
	private $id = 0;

	/**
	 * The attendee identifier that is associated with this record.
	 *
	 * @var integer
	 */
	private $attendee_id = 0;

	/**
	 * Holds a JSON blob of the changes for this attendee at a given date.
	 *
	 * @var string
	 */
	private $changes = '';

	/**
	 * The date that the changes occurred.
	 *
	 * @var date
	 */
	private $change_date = null;

	/**
	 * Constructor for the change log class.
	 *
	 * @param integer $id An existing record ID that we need to load.
	 */
	public function __construct( $id = 0 ) {
		$this->id = $id;
		$this->load();
	}

	/**
	 * Loads the change log record information.
	 */
	private function load() {
		global $wpdb;

		if ( 0 < $this->id ) {
			$sql       = 'SELECT attendeeID, changes, change_date FROM ' . PRO_ATTENDEE_CHANGELOG . ' WHERE id = %d';
			$changelog = $wpdb->get_row( $wpdb->prepare( $sql, $this->id ) );

			if ( false !== $changelog ) {
				$this->attendee_id = $changelog->attendeeID;
				$this->changes     = json_decode( stripslashes_deep( $changelog->changes ) );
				$this->change_date = date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $changelog->change_date ) );
			}
		}
	}

	/**
	 * Saves the change log information to the database.
	 */
	public function save() {
		global $wpdb;

		if ( 0 < $this->id ) {
			$wpdb->update(
				PRO_ATTENDEE_CHANGELOG,
				array(
					'attendeeID'  => $this->attendee_id,
					'changes'     => wp_json_encode( $this->changes ),
					'change_date' => date( 'Y-m-d H:i:s' ),
				),
				array( 'id' => $this->id ),
				array(
					'%d',
					'%s',
					'%s',
				),
				array( '%d' )
			);
		} else {
			$wpdb->insert(
				PRO_ATTENDEE_CHANGELOG,
				array(
					'attendeeID'  => $this->attendee_id,
					'changes'     => wp_json_encode( $this->changes ),
					'change_date' => date( 'Y-m-d H:i:s' ),
				),
				array(
					'%d',
					'%s',
					'%s',
				)
			);
		}
	}

	/**
	 * Deletes a change log entry from the database.
	 */
	public function delete() {
		global $wpdb;

		if ( 0 < $this->id ) {
			$wpdb->delete(
				PRO_ATTENDEE_CHANGELOG,
				array( 'id' => $this->id ),
				array( '%d' )
			);
		}
	}

	/**
	 * Loads the changelog entries for the attendee.
	 *
	 * @param  int $attendee_id The attendee ID that we want to load the changelog entries for.
	 *
	 * @return array            An array of Rsvp_Pro_Attendee_Changelog entries.
	 */
	public static function load_changelog_for_attendee( $attendee_id ) {
		global $wpdb;
		$changes = array();
		$sql     = 'SELECT id FROM ' . PRO_ATTENDEE_CHANGELOG . ' WHERE attendeeID = %d';
		$logs    = $wpdb->get_results( $wpdb->prepare( $sql, $attendee_id ) );
		foreach ( $logs as $l ) {
			$changes[] = new Rsvp_Pro_Attendee_Changelog( $l->id );
		}

		return $changes;
	}

	/**
	 * Gets the database identifier.
	 *
	 * @return int The database identifier for this record.
	 */
	public function get_id() {
		return $this->id;
	}

	/**
	 * Returns the date the change log entry was recorded.
	 *
	 * @return date The date when the change log entry was recorded.
	 */
	public function get_change_date() {
		return $this->change_date;
	}

	/**
	 * Holds an associated array of all the changes that occurred when this was recorded.
	 *
	 * @return array The associated array of changes.
	 */
	public function get_changes() {
		return $this->changes;
	}

	/**
	 * Takes an array of changes that needs to be recorded into the database.
	 *
	 * @param array $changes Associated array of field name and value of what was changed.
	 */
	public function set_changes( $changes ) {
		if ( ! is_array( $changes ) ) {
			$changes = array();
		}
		$this->changes = $changes;
	}

	/**
	 * Gets the attendee ID that is associated with this change log entry.
	 *
	 * @return int The attendee ID assocaited with this change log entry.
	 */
	public function get_attendee_id() {
		return $this->attendee_id;
	}

	/**
	 * Sets the attendee ID that is associated with this change log entry.
	 *
	 * @param int $attendee_id The attendee ID associated with this change log entry.
	 */
	public function set_attendee_id( $attendee_id ) {
		$this->attendee_id = $attendee_id;
	}
}
