<?php
/**
 * Class file that is wrapper for attendee event charges
 *
 * @since 4.3.6
 * @package rsvp-pro-plugin
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) :
	exit;
endif;

/**
 * Class for handling attendee custom question answers and storing it into the database.
 *
 * @since 3.8.8
 */
class Rsvp_Pro_Attendee_Event_Costs {

	/**
	 * The ID for this recorded answer.
	 *
	 * @var integer
	 */
	private $id = 0;

	/**
	 * The amount the attendee was charged.
	 *
	 * @var decimal
	 */
	private $amount = 0;

	/**
	 * The event associated with this cost.
	 *
	 * @var integer
	 */
	private $rsvp_event_id = 0;

	/**
	 * The attendee associated with this answer.
	 *
	 * @var integer
	 */
	private $attendee_id = 0;

	/**
	 * Constructor for this answer.
	 *
	 * @param integer $id The cost we wish to load.
	 */
	public function __construct( $id = 0 ) {
		$this->id = $id;
		$this->load();
	}

	/**
	 * Loads the information related to this attendee's event cost.
	 */
	private function load() {
		global $wpdb;

		$sql  = 'SELECT * FROM ' . PRO_ATTENDEE_EVENT_COSTS_TABLE . ' WHERE id = %d';
		$cost = $wpdb->get_row( $wpdb->prepare( $sql, $this->id ) );
		if ( null !== $cost ) {
			$this->attendee_id   = $cost->attendeeID;
			$this->amount        = $cost->amount;
			$this->rsvp_event_id = $cost->rsvpEventID;
		}
	}

	/**
	 * Deletes the attendee cost.
	 */
	public function delete() {
		global $wpdb;
		if ( 0 < $this->id ) {
			$wpdb->delete(
				PRO_ATTENDEE_EVENT_COSTS_TABLE,
				array( 'id' => $this->id ),
				array( '%d' )
			);
		}
	}

	/**
	 * Save the information to the database.
	 */
	public function save() {
		global $wpdb;

		if ( 0 < $this->id ) {
			$wpdb->update(
				PRO_ATTENDEE_EVENT_COSTS_TABLE,
				array(
					'attendeeID'  => $this->attendee_id,
					'rsvpEventID' => $this->rsvp_event_id,
					'amount'      => $this->amount,
				),
				array( 'id' => $this->id ),
				array(
					'%d',
					'%d',
					'%d',
				),
				array( '%d' )
			);
		} else {
			$wpdb->insert(
				PRO_ATTENDEE_EVENT_COSTS_TABLE,
				array(
					'attendeeID'  => $this->attendee_id,
					'rsvpEventID' => $this->rsvp_event_id,
					'amount'      => $this->amount,
				),
				array(
					'%d',
					'%d',
					'%d',
				)
			);
		}
	}

	/**
	 * Creates an array of all the event costs for an attendee.
	 *
	 * @param  int $attendee_id The id of the attendee we want the answers for.
	 *
	 * @return array            An array of Rsvp_Pro_Attendee_Event_Costs.
	 */
	public static function load_costs_for_attendee( $attendee_id ) {
		global $wpdb;

		$costs = array();
		$sql   = 'SELECT c.id FROM ' . PRO_ATTENDEE_EVENT_COSTS_TABLE . ' c 
			WHERE c.attendeeID = %d';
		$cst   = $wpdb->get_results( $wpdb->prepare( $sql, $attendee_id ) );
		foreach ( $cst as $c ) {
			$costs[] = new Rsvp_Pro_Attendee_Event_Costs( $c->id );
		}

		return $costs;
	}

	/**
	 * Returns the ID for this record.
	 *
	 * @return int The identifier for this record.
	 */
	public function get_id() {
		return $this->id;
	}

	/**
	 * The event ID that is associated with this cost.
	 *
	 * @return int The event identifier for this cost.
	 */
	public function get_rsvp_event_id() {
		return $this->rsvp_event_id;
	}

	/**
	 * Sets the rsvp event ID.
	 *
	 * @param int $event_id Sets the rsvp event ID for this attendee cost.
	 */
	public function set_rsvp_event_id( $event_id ) {
		if ( is_numeric( $event_id ) && ( 0 < $event_id ) ) {
			$this->rsvp_event_id = $event_id;
		} else {
			$this->rsvp_event_id = 0;
		}

	}

	/**
	 * Returns the attendee ID associated with the cost.
	 *
	 * @return integer The attendee ID.
	 */
	public function get_attendee_id() {
		return $this->attendee_id;
	}

	/**
	 * Sets the attendee that should be associated with this event cost.
	 *
	 * @param integer $attendee_id The attendee ID we want associated with this cost.
	 */
	public function set_attendee_id( $attendee_id ) {
		if ( is_numeric( $attendee_id ) && ( 0 <= $attendee_id ) ) {
			$this->attendee_id = $attendee_id;
		} else {
			$this->attendee_id = 0;
		}
	}

	/**
	 * Gets the amount the attendee was charged for this event.
	 *
	 * @return string The amount the attendee was charged.
	 */
	public function get_amount() {
		return $this->amount;
	}

	/**
	 * Records the amount the attendee was charged for the event.
	 *
	 * @param string $amount The amount that the user was charged for the event.
	 */
	public function set_amount( $amount ) {
		$this->amount = $amount;
	}
}
