<?php
/**
 * Class file that is wrapper for attendee sub-event information
 *
 * @since 3.8.8
 * @package rsvp-pro-plugin
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) :
	exit;
endif;

/**
 * Class for handling attendee sub-event information and storing it into the database.
 *
 * @since 3.8.8
 */
class Rsvp_Pro_Attendee_Subevent {

	/**
	 * The database identifier for the sub-event record.
	 *
	 * @var integer
	 */
	private $id = 0;

	/**
	 * The RSVP date for this entry.
	 *
	 * @var string
	 */
	private $rsvp_date = null;

	/**
	 * The RSVP status for this sub-event.
	 *
	 * @var string
	 */
	private $rsvp_status = 'NoResponse';

	/**
	 * The event database identifier associated with this record.
	 *
	 * @var integer
	 */
	private $event_id = 0;

	/**
	 * The attendee ID associated with this record.
	 *
	 * @var integer
	 */
	private $attendee_id = 0;

	/**
	 * If the attendee has checked in to the event or not.
	 *
	 * @var string
	 */
	private $checked_in = 'N';

	/**
	 * The attendee unique token.
	 *
	 * @var string
	 */
	private $token = '';

	/**
	 * Holds all of the answers for custom questions that this attendee has added.
	 *
	 * @var array
	 */
	private $custom_question_answers = array();

	/**
	 * The number of additional guests that this person can invite.
	 *
	 * @var integer
	 */
	private $num_guests = null;

	/**
	 * The amount the attendee has paid for the event.
	 *
	 * @var string
	 */
	private $amount = '';

	/**
	 * The payment status for this attendee.
	 *
	 * @var string
	 */
	private $payment_status = '';

	/**
	 * The transaction id related to the payment for this attendee.
	 *
	 * @var string
	 */
	private $transaction_id = '';

	/**
	 * Constructor for this record.
	 *
	 * @param integer $id The database identifier that we want to load the information from.
	 */
	public function __construct( $id = 0 ) {
		$this->id = $id;
		$this->load();
	}

	/**
	 * Loads the information for this sub-event database record.
	 */
	private function load() {
		global $wpdb;

		if ( 0 < $this->id ) {
			$sql      = 'SELECT rsvpDate, rsvpStatus, rsvpEventID, rsvpAttendeeID, 
				checked_in, token, numGuests, amount, payment_status, transaction_id  
				FROM ' . PRO_ATTENDEE_SUB_EVENTS_TABLE . ' WHERE id = %d';
			$attendee = $wpdb->get_row( $wpdb->prepare( $sql, $this->id ) );

			if ( false !== $attendee ) {
				$this->rsvp_date      = ( null === $attendee->rsvpDate ) ? null : date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $attendee->rsvpDate ) );
				$this->rsvp_status    = rsvp_pro_format_rsvp_status_for_database( $attendee->rsvpStatus );
				$this->event_id       = $attendee->rsvpEventID;
				$this->attendee_id    = $attendee->rsvpAttendeeID;
				$this->checked_in     = $attendee->checked_in;
				$this->token          = stripslashes_deep( $attendee->token );
				$this->num_guests     = stripslashes_deep( $attendee->numGuests );
				$this->amount         = stripslashes_deep( $attendee->amount );
				$this->payment_status = stripslashes_deep( $attendee->payment_status );
				$this->transaction_id = stripslashes_deep( $attendee->transaction_id );
			}

			$this->custom_question_answers = Rsvp_Pro_Attendee_Answers::load_answers_for_attendee( $this->id, $this->event_id );
		}
	}

	/**
	 * Saves the sub-event information to the database.
	 */
	public function save() {
		global $wpdb;

		if ( 0 < $this->id ) {
			$wpdb->update(
				PRO_ATTENDEE_SUB_EVENTS_TABLE,
				array(
					'rsvpDate'       => date( 'Y-m-d H:i:s', strtotime( $this->rsvp_date ) ),
					'rsvpStatus'     => rsvp_pro_format_rsvp_status_for_database( $this->rsvp_status ),
					'rsvpEventID'    => $this->event_id,
					'rsvpAttendeeID' => $this->attendee_id,
					'checked_in'     => $this->checked_in,
					'numGuests'      => $this->num_guests,
					'amount'         => $this->amount,
					'payment_status' => $this->payment_status,
					'transaction_id' => $this->transaction_id,
				),
				array( 'id' => $this->id ),
				array(
					'%s', // rsvpDate.
					'%s', // rsvpStatus.
					'%d', // rsvpEventID.
					'%d', // rsvpAttendeeID.
					'%s', // checked_in.
					'%d', // numGuests.
					'%d', // amount.
					'%s', // payment_status.
					'%s', // transaction_id.
				),
				array( '%d' )
			);
		} else {
			$wpdb->insert(
				PRO_ATTENDEE_SUB_EVENTS_TABLE,
				array(
					'rsvpDate'       => date( 'Y-m-d H:i:s', strtotime( $this->rsvp_date ) ),
					'rsvpStatus'     => rsvp_pro_format_rsvp_status_for_database( $this->rsvp_status ),
					'rsvpEventID'    => $this->event_id,
					'rsvpAttendeeID' => $this->attendee_id,
					'checked_in'     => $this->checked_in,
					'token'          => rsvp_pro_generate_attendee_token(),
					'numGuests'      => $this->num_guests,
					'amount'         => $this->amount,
					'payment_status' => $this->payment_status,
					'transaction_id' => $this->transaction_id,
				),
				array(
					'%s', // rsvpDate.
					'%s', // rsvpStatus.
					'%d', // rsvpEventID.
					'%d', // rsvpAttendeeID.
					'%s', // checked_in.
					'%s', // token.
					'%d', // numGuests.
					'%d', // amount.
					'%s', // payment_status.
					'%s', // transaction_id.
				)
			);
		}
	}

	/**
	 * Deletes the sub-event information to the database.
	 */
	public function delete() {
		global $wpdb;

		if ( 0 < $this->id ) {
			$wpdb->delete(
				PRO_ATTENDEE_SUB_EVENTS_TABLE,
				array( 'id' => $this->id ),
				array( '%d' )
			);
		}
	}

	/**
	 * Gets the sub-event information for a specific attendee.
	 *
	 * @param  int $attendee_id The attendee ID that we want to get the sub-event information for.
	 *
	 * @return array              Array of Rsvp_Pro_Attendee_Subevent for each sub-event.
	 */
	public static function load_subevents_for_attendee( $attendee_id ) {
		global $wpdb;

		$subevents     = array();
		$sql           = 'SELECT id FROM ' . PRO_ATTENDEE_SUB_EVENTS_TABLE . ' WHERE rsvpAttendeeID = %d';
		$subevent_info = $wpdb->get_results( $wpdb->prepare( $sql, $attendee_id ) );
		foreach ( $subevent_info as $se ) {
			$subevents[] = new Rsvp_Pro_Attendee_Subevent( $se->id );
		}

		return $subevents;
	}

	/**
	 * Tries to load a specific sub-event for an attendee and if no record exists a new object is returned.
	 *
	 * @param  int $event_id    The event ID of a sub or a top-level event we want to try to load for.
	 * @param  int $attendee_id The attendee ID we want the record for.
	 * @return object           A Rsvp_Pro_Attendee_Subevent object is returned.
	 */
	public static function load_subevent_for_attendee( $event_id, $attendee_id ) {
		global $wpdb;

		$sql    = 'SELECT id FROM ' . PRO_ATTENDEE_SUB_EVENTS_TABLE . ' WHERE rsvpAttendeeID = %d AND rsvpEventID = %d';
		$record = $wpdb->get_row( $wpdb->prepare( $sql, $attendee_id, $event_id ) );
		if ( $record === null ) {
			$se_record = new Rsvp_Pro_Attendee_Subevent();
			$se_record->set_attendee_id( $attendee_id );
			$se_record->set_event_id( $event_id );
			return $se_record;
		} else {
			return new Rsvp_Pro_Attendee_Subevent( $record->id );
		}
	}

	/**
	 * Gets the database identifier for this sub-event record.
	 *
	 * @return int The database identifier for this record.
	 */
	public function get_id() {
		return $this->id;
	}

	/**
	 * Gets the unique token for the sub-event record.
	 *
	 * @return string The token for this sub-event record.
	 */
	public function get_token() {
		return $this->token;
	}

	/**
	 * Returns the RSVP date for this sub-event.
	 *
	 * @return date The RSVP date formatted in the current WordPress instance date format.
	 */
	public function get_rsvp_date() {
		return $this->rsvp_date;
	}

	/**
	 * Sets the RSVP date for this sub-event.
	 *
	 * @param string $rsvp_date Holding a date in string format.
	 */
	public function set_rsvp_date( $rsvp_date ) {
		if ( false === strtotime( $rsvp_date ) ) {
			$rsvp_date = null;
		}

		$this->rsvp_date = $rsvp_date;
	}

	/**
	 * The RSVP status for this sub-event.
	 *
	 * @return string The RSVP status for this sub-event.
	 */
	public function get_rsvp_status() {
		return $this->rsvp_status;
	}

	/**
	 * Sets the RSVP status for this sub-event.
	 *
	 * @param string $status The RSVP status for this sub-event.
	 */
	public function set_rsvp_status( $status ) {
		$this->rsvp_status = rsvp_pro_format_rsvp_status_for_database( $status );
	}

	/**
	 * Returns the event ID for this sub-event.
	 *
	 * @return int The event ID related to this sub-event.
	 */
	public function get_event_id() {
		return $this->event_id;
	}

	/**
	 * Sets the event ID for this sub-event.
	 *
	 * @param int $event_id The event ID for this sub-event.
	 */
	public function set_event_id( $event_id ) {
		if ( ! is_numeric( $event_id ) || ( 0 >= $event_id ) ) {
			$event_id = 0;
		}

		$this->event_id = $event_id;
	}

	/**
	 * Gets the attendee ID for this sub-event.
	 *
	 * @return int The attendee ID for this sub-event.
	 */
	public function get_attendee_id() {
		return $this->attendee_id;
	}

	/**
	 * Sets the attendee ID for this sub-event.
	 *
	 * @param int $attendee_id The attendee ID for the sub-event.
	 */
	public function set_attendee_id( $attendee_id ) {
		if ( ! is_numeric( $attendee_id ) || ( 0 >= $attendee_id ) ) {
			$attendee_id = 0;
		}

		$this->attendee_id = $attendee_id;
	}

	/**
	 * The check-in field for the sub-event.
	 *
	 * @return string Either a 'Y' or 'N' if the people are checked-in.
	 */
	public function get_checked_in() {
		return $this->checked_in;
	}

	/**
	 * Sets the check-in field for the sub-event.
	 *
	 * @param string $checked_in Either 'Y' or 'N' for the checked-in value.
	 */
	public function set_checked_in( $checked_in ) {
		if ( 'Y' === strtoupper( $checked_in ) ) {
			$this->checked_in = 'Y';
		} else {
			$this->checked_in = 'N';
		}
	}

	/**
	 * Gets all of the custom question answers for an attendee.
	 *
	 * @return array An array of Rsvp_Pro_Attendee_Answers.
	 */
	public function get_custom_question_answers() {
		return $this->custom_question_answers;
	}

	/**
	 * Gets how many number of guests the person can bring.
	 *
	 * @return int The number of guests the person can bring.
	 */
	public function get_num_guests() {
		return $this->num_guests;
	}

	/**
	 * Sets the number of guests the person can bring.
	 *
	 * @param int $num_guests The number of guests that a person can bring.
	 */
	public function set_num_guests( $num_guests ) {
		if ( ! is_numeric( $num_guests ) || ( $num_guests < 0 ) ) {
			$num_guests = null;
		}

		$this->num_guests = $num_guests;
	}

	/**
	 * Gets the amount the attendee paid for this event.
	 *
	 * @return string The amount paid, this should usually be numeric.
	 */
	public function get_amount() {
		return $this->amount;
	}

	/**
	 * Sets the amount that the attendee paid for the event.
	 *
	 * @param string $in The amount the person paid for the event.
	 */
	public function set_amount( $in ) {
		$this->amount = $in;
	}

	/**
	 * Gets the current payment status for the event attendee combination.
	 *
	 * @return string The payment status for the attendee for this event.
	 */
	public function get_payment_status() {
		return $this->payment_status;
	}

	/**
	 * Sets the payment status for this attendee for this event.
	 *
	 * @param string $in The payment status.
	 */
	public function set_payment_status( $in ) {
		$this->payment_status = $in;
	}

	/**
	 * Returns the last transaction id for this attendee for the event.
	 *
	 * @return string The last transaction id for this attendee for the event.
	 */
	public function get_transaction_id() {
		return $this->transaction_id;
	}

	/**
	 * Sets the transaction id for the last transaction related to this attendee for this event.
	 *
	 * @param string $in The transaction id.
	 */
	public function set_transaction_id( $in ) {
		$this->transaction_id = $in;
	}

	/**
	 * Adds a custom question answer for an attendee.
	 *
	 * @param Rsvp_Pro_Attendee_Answers $custom_question_answer The custom question answer we want to save for an attendee.
	 */
	public function add_custom_question_answer( $custom_question_answer ) {
		if ( is_a( $custom_question_answer, 'Rsvp_Pro_Attendee_Answers' ) ) {
			for ( $i = 0; $i < count( $this->custom_question_answers ); $i ++ ) {
				if ( $this->custom_question_answers[ $i ]->get_id() === $custom_question_answer->get_id() ) {
					$this->custom_question_answers[ $i ] = $custom_question_answer;

					return;
				}
			}

			$this->custom_question_answers[] = $custom_question_answer;
		}
	}
}
