<?php
/**
 * Class file that is wrapper for attendees
 *
 * @since 3.8.8
 * @package rsvp-pro-plugin
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) :
	exit;
endif;

/**
 * Class for handling attendee management and storing it into the database.
 *
 * @since 3.8.8
 */
class Rsvp_Pro_Attendee {

	/**
	 * The attendee ID.
	 *
	 * @var integer
	 */
	private $id = 0;

	/**
	 * The attendee's first name.
	 *
	 * @var string
	 */
	private $first_name = '';

	/**
	 * The attendee's last name.
	 *
	 * @var string
	 */
	private $last_name = '';

	/**
	 * The date and time of the attendee last RSVP'd.
	 *
	 * @var date
	 */
	private $rsvp_date = null;

	/**
	 * The current attendee RSVP Status.
	 *
	 * @var string
	 */
	private $rsvp_status = 'NoResponse';

	/**
	 * The note for the attendee.
	 *
	 * @var string
	 */
	private $note = '';

	/**
	 * If this attendee is an additional attendee.
	 *
	 * @var string
	 */
	private $additional_attendee = 'N';

	/**
	 * The personal greeting for the attendee.
	 *
	 * @var string
	 */
	private $personal_greeting = '';

	/**
	 * The attendee's email.
	 *
	 * @var string
	 */
	private $email = '';

	/**
	 * The attendee's passcode.
	 *
	 * @var string
	 */
	private $passcode = '';

	/**
	 * The RSVP event ID that is associated with this attendee.
	 *
	 * @var integer
	 */
	private $rsvp_event_id = 0;

	/**
	 * The number of additional guests that this person can invite.
	 *
	 * @var integer
	 */
	private $num_guests = null;

	/**
	 * The suffix for this attendee.
	 *
	 * @var string
	 */
	private $suffix = '';

	/**
	 * The salutation for this attendee.
	 *
	 * @var string
	 */
	private $salutation = '';

	/**
	 * The nick names for this attendee.
	 *
	 * @var string
	 */
	private $nicknames = '';

	/**
	 * If this attendeeis a primary attendee or not.
	 *
	 * @var string
	 */
	private $primary_attendee = '';

	/**
	 * If this attendee is checked in or not.
	 *
	 * @var string
	 */
	private $checked_in = 'N';

	/**
	 * The group ID for this attendee.
	 *
	 * @var string
	 */
	private $group_id = '';

	/**
	 * A unique token for this attendee.
	 *
	 * @var string
	 */
	private $token = '';

	/**
	 * The amount the attendee has paid for the event.
	 *
	 * @var string
	 */
	private $amount = '';

	/**
	 * The payment status for this attendee.
	 *
	 * @var string
	 */
	private $payment_status = '';

	/**
	 * The transaction id related to the payment for this attendee.
	 *
	 * @var string
	 */
	private $transaction_id = '';

	/**
	 * The alternate passcode for an attendee.
	 *
	 * @var string
	 */
	private $alternate_passcode = '';

	/**
	 * Any associated attendees for this attendee.
	 *
	 * @var array of Ints holding associated attendee IDs.
	 */
	private $associated_attendees = array();

	/**
	 * Holds all of the answers for custom questions that this attendee has added.
	 *
	 * @var array
	 */
	private $custom_question_answers = array();

	/**
	 * Holds all of the attendee's responses for sub-events.
	 *
	 * @var array
	 */
	private $sub_event_responses = array();

	/**
	 * Holds all of the attendee's event costs.
	 *
	 * @var array
	 */
	private $event_costs = array();

	/**
	 * Holds the change log for this attendee.
	 *
	 * @var array
	 */
	private $changelog = array();

	/**
	 * Loads an attendee or an empty object for attendees.
	 *
	 * @param integer $attendee_id The attendee ID that we want to load.
	 */
	public function __construct( $attendee_id = 0 ) {
		if ( is_numeric( $attendee_id ) && ( $attendee_id > 0 ) ) {
			$this->id = $attendee_id;
			$this->load();
		}
	}

	/**
	 * Loads the information for this attendee.
	 *
	 * Postcondition: Attendee information has been loaded if the attendee ID can be found.
	 */
	private function load() {
		global $wpdb;

		if ( $this->id > 0 ) {
			$sql      = 'SELECT * FROM ' . PRO_ATTENDEES_TABLE . ' WHERE id = %d';
			$attendee = $wpdb->get_row( $wpdb->prepare( $sql, $this->id ) );

			if ( false !== $attendee ) {
				$this->first_name          = stripslashes_deep( $attendee->firstName );
				$this->last_name           = stripslashes_deep( $attendee->lastName );
				$this->rsvp_date           = ( null === $attendee->rsvpDate ) ? null : date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $attendee->rsvpDate ) );
				$this->rsvp_status         = stripslashes_deep( $attendee->rsvpStatus );
				$this->note                = stripslashes_deep( $attendee->note );
				$this->additional_attendee = ( 'Y' === strtoupper( $attendee->additionalAttendee ) ) ? 'Y' : 'N';
				$this->personal_greeting   = stripslashes_deep( $attendee->personalGreeting );
				$this->email               = stripslashes_deep( $attendee->email );
				$this->passcode            = stripslashes_deep( $attendee->passcode );
				$this->alternate_passcode  = stripslashes_deep( $attendee->alternate_passcode );
				$this->rsvp_event_id       = $attendee->rsvpEventID;
				$this->num_guests          = stripslashes_deep( $attendee->numGuests );
				$this->suffix              = stripslashes_deep( $attendee->suffix );
				$this->salutation          = stripslashes_deep( $attendee->salutation );
				$this->nicknames           = stripslashes_deep( $attendee->nicknames );
				$this->primary_attendee    = ( 'Y' === strtoupper( $attendee->primaryAttendee ) ) ? 'Y' : 'N';
				$this->checked_in          = ( 'Y' === strtoupper( $attendee->checked_in ) ) ? 'Y' : 'N';
				$this->group_id            = stripslashes_deep( $attendee->group_id );
				$this->token               = stripslashes_deep( $attendee->token );
				$this->amount              = stripslashes_deep( $attendee->amount );
				$this->payment_status      = stripslashes_deep( $attendee->amount );
				$this->transaction_id      = stripslashes_deep( $attendee->transaction_id );

				$sql                  = 'SELECT associatedAttendeeID FROM ' . PRO_ASSOCIATED_ATTENDEES_TABLE . ' WHERE attendeeID = %d';
				$associated_attendees = $wpdb->get_results( $wpdb->prepare( $sql, $this->id ) );
				foreach ( $associated_attendees as $aa ) {
					$this->associated_attendees[] = $aa->associatedAttendeeID;
				}

				$this->custom_question_answers = Rsvp_Pro_Attendee_Answers::load_answers_for_attendee( $this->id, $this->rsvp_event_id );
				$this->sub_event_responses     = Rsvp_Pro_Attendee_Subevent::load_subevents_for_attendee( $this->id );
				$this->changelog               = Rsvp_Pro_Attendee_Changelog::load_changelog_for_attendee( $this->id );
				$this->event_costs             = Rsvp_Pro_Attendee_Event_Costs::load_costs_for_attendee( $this->id );
			}
		}
	}

	/**
	 * Saves the attendee information.
	 */
	public function save() {
		global $wpdb;

		if ( 0 >= $this->id ) {
			$wpdb->insert(
				PRO_ATTENDEES_TABLE,
				array(
					'rsvpEventID' => $this->rsvp_event_id,
					'firstName'   => $this->first_name,
					'lastName'    => $this->last_name,
					'token'       => rsvp_pro_generate_attendee_token(),
				),
				array(
					'%d',
					'%s',
					'%s',
					'%s',
				)
			);

			$this->id = $wpdb->insert_id;
		}

		$wpdb->update(
			PRO_ATTENDEES_TABLE,
			array(
				'firstName'          => $this->first_name,
				'lastName'           => $this->last_name,
				'rsvpDate'           => ( null !== $this->rsvp_date  ) ? wp_date( 'Y-m-d H:i:s', strtotime( $this->rsvp_date ) ) : null,
				'rsvpStatus'         => rsvp_pro_format_rsvp_status_for_database( $this->rsvp_status ),
				'note'               => $this->note,
				'additionalAttendee' => ( 'Y' === strtoupper( $this->additional_attendee ) ) ? 'Y' : 'N',
				'personalGreeting'   => $this->personal_greeting,
				'email'              => $this->email,
				'passcode'           => $this->passcode,
				'alternate_passcode' => $this->alternate_passcode,
				'rsvpEventID'        => $this->rsvp_event_id,
				'numGuests'          => $this->num_guests,
				'suffix'             => $this->suffix,
				'salutation'         => $this->salutation,
				'nicknames'          => $this->nicknames,
				'primaryAttendee'    => ( 'Y' === strtoupper( $this->primary_attendee ) ) ? 'Y' : 'N',
				'checked_in'         => ( 'Y' === strtoupper( $this->checked_in ) ) ? 'Y' : 'N',
				'group_id'           => $this->group_id,
				'amount'             => $this->amount,
				'payment_status'     => $this->payment_status,
				'transaction_id'     => $this->transaction_id,
			),
			array( 'id' => $this->id ),
			array(
				'%s', // firstName.
				'%s', // lastName.
				'%s', // rsvpDate.
				'%s', // rsvpStatus.
				'%s', // note.
				'%s', // additionalAttendee.
				'%s', // personalGreeting.
				'%s', // email.
				'%s', // passcode.
				'%s', // alternate_passcode.
				'%d', // rsvpEventID.
				'%d', // numGuests.
				'%s', // suffix.
				'%s', // salutation.
				'%s', // nicknames.
				'%s', // primaryAttendee.
				'%s', // checked_in.
				'%s', // group_id.
				'%d', // amount.
				'%s', // payment_status.
				'%s', // transaction_id.
			),
			array( '%d' )
		);

		foreach ( $this->custom_question_answers as $a ) {
			$a->set_attendee_id( $this->id );
			$a->save();
		}

		foreach ( $this->sub_event_responses as $se ) {
			$se->set_attendee_id( $this->id );
			$se->save();
		}

		foreach ( $this->changelog as $c ) {
			$c->set_attendee_id( $this->id );
			$c->save();
		}

		foreach( $this->event_costs as $ec ) {
			$ec->set_attendee_id( $this->id );
			$ec->save();
		}

		$this->save_associated_attendees();
	}

	/**
	 * Saves the associated attendees to the database.
	 */
	private function save_associated_attendees() {
		global $wpdb;

		$wpdb->delete(
			PRO_ASSOCIATED_ATTENDEES_TABLE,
			array( 'attendeeID' => $this->id ),
			array( '%d' )
		);

		$wpdb->delete(
			PRO_ASSOCIATED_ATTENDEES_TABLE,
			array( 'associatedAttendeeID' => $this->id ),
			array( '%d' )
		);

		foreach ( $this->associated_attendees as $aa ) {
			$wpdb->insert(
				PRO_ASSOCIATED_ATTENDEES_TABLE,
				array(
					'attendeeID'           => $this->id,
					'associatedAttendeeID' => $aa,
				),
				array(
					'%d',
					'%d',
				)
			);

			$wpdb->insert(
				PRO_ASSOCIATED_ATTENDEES_TABLE,
				array(
					'attendeeID'           => $aa,
					'associatedAttendeeID' => $this->id,
				),
				array(
					'%d',
					'%d',
				)
			);
		}
	}

	/**
	 * Deletes the attendee and related information.
	 */
	public function delete() {
		global $wpdb;

		if ( 0 < $this->id ) {

			foreach ( $this->custom_question_answers as $a ) {
				$a->delete();
			}

			foreach ( $this->sub_event_responses as $se ) {
				$se->delete();
			}

			foreach ( $this->changelog as $c ) {
				$c->delete();
			}

			foreach ( $this->event_costs as $ec ) {
				$ec->delete();
			}

			$wpdb->delete(
				PRO_EVENT_ATTENDEE_TABLE,
				array( 'rsvpAttendeeID' => $this->id ),
				array( '%d' )
			);

			$wpdb->delete(
				PRO_QUESTION_ATTENDEES_TABLE,
				array( 'attendeeID' => $this->id ),
				array( '%d' )
			);

			$wpdb->delete(
				PRO_ASSOCIATED_ATTENDEES_TABLE,
				array( 'attendeeID' => $this->id ),
				array( '%d' )
			);

			$wpdb->delete(
				PRO_ASSOCIATED_ATTENDEES_TABLE,
				array( 'associatedAttendeeID' => $this->id ),
				array( '%d' )
			);

			$wpdb->delete(
				PRO_ATTENDEES_TABLE,
				array( 'id' => $this->id ),
				array( '%d' )
			);
		}
	}

	/**
	 * Returns the ID for the attendee record.
	 *
	 * @return int The database row ID.
	 */
	public function get_id() {
		return $this->id;
	}

	/**
	 * Returns a string that uniquely identifies the attendee.
	 *
	 * @return string The token that uniquely identifies the attendee.
	 */
	public function get_token() {
		return $this->token;
	}

	/**
	 * The attendee's first name.
	 *
	 * @return string The attendee's first name.
	 */
	public function get_first_name() {
		return $this->first_name;
	}

	/**
	 * Sets the attendee's first name.
	 *
	 * @param string $first_name The attendee's first name.
	 */
	public function set_first_name( $first_name ) {
		$this->first_name = $first_name;
	}

	/**
	 * The attendee's last name.
	 *
	 * @return string The attendee's last name.
	 */
	public function get_last_name() {
		return $this->last_name;
	}

	/**
	 * Sets the attendee's last name.
	 *
	 * @param string $last_name The attendee's last name.
	 */
	public function set_last_name( $last_name ) {
		$this->last_name = $last_name;
	}

	/**
	 * Gets the date of when the attendee RSVP'd.
	 *
	 * @return date The date when the attendee RSVP'd.
	 */
	public function get_rsvp_date() {
		return $this->rsvp_date;
	}

	/**
	 * Sets the RSVP date.
	 *
	 * @param date $rsvp_date The RSVP date that the person has rsvp'd with.
	 */
	public function set_rsvp_date( $rsvp_date ) {
		if ( false === strtotime( $rsvp_date ) ) {
			$rsvp_date = null;
		}

		$this->rsvp_date = $rsvp_date;
	}

	/**
	 * Gets the RSVP status for an attendee.
	 *
	 * @return string The RSVP status for an attendee.
	 */
	public function get_rsvp_status() {
		return $this->rsvp_status;
	}

	/**
	 * Sets the RSVP status for an attendee.
	 *
	 * @param string $rsvp_status The RSVP status for the attendee.
	 */
	public function set_rsvp_status( $rsvp_status ) {
		$this->rsvp_status = rsvp_pro_format_rsvp_status_for_database( $rsvp_status );
	}

	/**
	 * Gets the note for the attendee.
	 *
	 * @return string The attendee's note.
	 */
	public function get_note() {
		return $this->note;
	}

	/**
	 * Sets the attendee's note.
	 *
	 * @param string $note The attendee's note.
	 */
	public function set_note( $note ) {
		$this->note = $note;
	}

	/**
	 * Gets if the person is an additional attendee.
	 *
	 * @return string If the person is an additional attendee.
	 */
	public function get_additional_attendee() {
		return $this->additional_attendee;
	}

	/**
	 * Sets if the person is an additional attendee.
	 *
	 * @param string $additional Specifies if the person is an additional attendee.
	 */
	public function set_additional_attendee( $additional ) {
		if ( strtoupper( $additional ) !== 'Y' ) {
			$additional = 'N';
		}

		$this->additional_attendee = $additional;
	}

	/**
	 * Gets the attendee's personal greeting.
	 *
	 * @return string The attendee's personal greeting.
	 */
	public function get_personal_greeting() {
		return $this->personal_greeting;
	}

	/**
	 * Sets the attendee's personal greeting.
	 *
	 * @param string $greeting The attendee's personal greeting.
	 */
	public function set_personal_greeting( $greeting ) {
		$this->personal_greeting = $greeting;
	}

	/**
	 * Gets the attendee's email address.
	 *
	 * @return string The attendee's email address.
	 */
	public function get_email() {
		return $this->email;
	}

	/**
	 * Sets the attendee's email address.
	 *
	 * @param string $email The attendee's email address.
	 */
	public function set_email( $email ) {
		$this->email = $email;
	}

	/**
	 * Gets the passcode for the attendee.
	 *
	 * @return string The attendee's passcode.
	 */
	public function get_passcode() {
		return $this->passcode;
	}

	/**
	 * Sets the attendee's passcode.
	 *
	 * @param string $passcode The attendee's passcode.
	 */
	public function set_passcode( $passcode ) {
		$this->passcode = $passcode;
	}

	/**
	 * Returns the alternate passcode for an attendee.
	 *
	 * @return string The attendee's alternate passcode.
	 */
	public function get_alternate_passcode() {
		return $this->alternate_passcode;
	}

	/**
	 * Sets the attendee's alternate passcode.
	 *
	 * @param string $passcode The attendee's alternate passcode.
	 */
	public function set_alternate_passcode( $passcode ) {
		$this->alternate_passcode = $passcode;
	}

	/**
	 * Gets the RSVP event ID.
	 *
	 * @return int The RSVP event ID.
	 */
	public function get_rsvp_event_id() {
		return $this->rsvp_event_id;
	}

	/**
	 * Sets the RSVP event ID.
	 *
	 * @param int $event_id The RSVP event ID.
	 */
	public function set_rsvp_event_id( $event_id ) {
		$this->rsvp_event_id = $event_id;
	}

	/**
	 * Gets how many number of guests the person can bring.
	 *
	 * @return int The number of guests the person can bring.
	 */
	public function get_num_guests() {
		return $this->num_guests;
	}

	/**
	 * Sets the number of guests the person can bring.
	 *
	 * @param int $num_guests The number of guests that a person can bring.
	 */
	public function set_num_guests( $num_guests ) {
		if ( ! is_numeric( $num_guests ) || ( $num_guests < 0 ) ) {
			$num_guests = null;
		}

		$this->num_guests = $num_guests;
	}

	/**
	 * Gets the attendee's suffix.
	 *
	 * @return string The attendee's suffix.
	 */
	public function get_suffix() {
		return $this->suffix;
	}

	/**
	 * Sets the attendee's suffix.
	 *
	 * @param string $suffix The attendee's suffix.
	 */
	public function set_suffix( $suffix ) {
		$this->suffix = $suffix;
	}

	/**
	 * Gets the attendee's salutation.
	 *
	 * @return string The attendee's salutation.
	 */
	public function get_salutation() {
		return $this->salutation;
	}

	/**
	 * Sets the attendee's salutation.
	 *
	 * @param string $salutation The attendee's salutation.
	 */
	public function set_salutation( $salutation ) {
		$this->salutation = $salutation;
	}

	/**
	 * Gets the attendee's nick name.
	 *
	 * @return string The attendee's nick name.
	 */
	public function get_nicknames() {
		return $this->nicknames;
	}

	/**
	 * Sets the attendee's nick name.
	 *
	 * @param string $nicknames The attendee's nick name.
	 */
	public function set_nicknames( $nicknames ) {
		$this->nicknames = $nicknames;
	}

	/**
	 * Gets the attendee's primary attendee.
	 *
	 * @return string If the attendee is a primary attendee.
	 */
	public function get_primary_attendee() {
		return $this->primary_attendee;
	}

	/**
	 * Sets if the attendee is a primary attendee or not.
	 *
	 * @param string $primary_attendee If the attendee is a primary attendee or not.
	 */
	public function set_primary_attendee( $primary_attendee ) {
		$primary_attendee = strtoupper( $primary_attendee );
		if ( ( 'Y' !== $primary_attendee ) && ( 'N' !== $primary_attendee ) ) {
			$primary_attendee = '';
		}

		$this->primary_attendee = $primary_attendee;
	}

	/**
	 * Gets if the attendee is checked in.
	 *
	 * @return string If the attendee is checked in or not.
	 */
	public function get_checked_in() {
		return $this->checked_in;
	}

	/**
	 * Sets if the attendee is checked in or not.
	 *
	 * @param string $checked_in If the attendee is checked in or not Y or N are valid values.
	 */
	public function set_checked_in( $checked_in ) {
		if ( 'Y' !== strtoupper( $checked_in ) ) {
			$checked_in = 'N';
		}

		$this->checked_in = $checked_in;
	}

	/**
	 * Gets the attendee's group ID.
	 *
	 * @return string The attendee's group id.
	 */
	public function get_group_id() {
		return $this->group_id;
	}

	/**
	 * Sets the attendee's group id.
	 *
	 * @param string $group_id The attendee's group id.
	 */
	public function set_group_id( $group_id ) {
		$this->group_id = $group_id;
	}

	/**
	 * Gets the amount the attendee paid for this event.
	 *
	 * @return string The amount paid, this should usually be numeric.
	 */
	public function get_amount() {
		return $this->amount;
	}

	/**
	 * Sets the amount that the attendee paid for the event.
	 *
	 * @param string $in The amount the person paid for the event.
	 */
	public function set_amount( $in ) {
		$this->amount = $in;
	}

	/**
	 * Gets the current payment status for the event attendee combination.
	 *
	 * @return string The payment status for the attendee for this event.
	 */
	public function get_payment_status() {
		return $this->payment_status;
	}

	/**
	 * Sets the payment status for this attendee for this event.
	 *
	 * @param string $in The payment status.
	 */
	public function set_payment_status( $in ) {
		$this->payment_status = $in;
	}

	/**
	 * Returns the last transaction id for this attendee for the event.
	 *
	 * @return string The last transaction id for this attendee for the event.
	 */
	public function get_transaction_id() {
		return $this->transaction_id;
	}

	/**
	 * Sets the transaction id for the last transaction related to this attendee for this event.
	 *
	 * @param string $in The transaction id.
	 */
	public function set_transaction_id( $in ) {
		$this->transaction_id = $in;
	}

	/**
	 * Gets all of the associated attendees.
	 *
	 * @return array Array of Ints that are associated attendee IDs.
	 */
	public function get_associated_attendees() {
		return $this->associated_attendees;
	}

	/**
	 * Sets associated attendees for this attendee.
	 *
	 * @param array $attendees Array of integers of associated attendees.
	 */
	public function set_associated_attendees( $attendees ) {
		if ( is_array( $attendees ) ) {
			$sanitized_attendees = array();

			foreach ( $attendees as $a ) {
				if ( is_numeric( $a ) && ( $a > 0 ) ) {
					$sanitized_attendees[] = $a;
				}
			}

			$this->associated_attendees = $sanitized_attendees;
		}
	}

	/**
	 * Gets all of the custom question answers for an attendee.
	 *
	 * @return array An array of Rsvp_Pro_Attendee_Answers.
	 */
	public function get_custom_question_answers() {
		return $this->custom_question_answers;
	}

	/**
	 * Adds a custom question answer for an attendee.
	 *
	 * @param Rsvp_Pro_Attendee_Answers $custom_question_answer The custom question answer we want to save for an attendee.
	 */
	public function add_custom_question_answer( $custom_question_answer ) {
		if ( is_a( $custom_question_answer, 'Rsvp_Pro_Attendee_Answers' ) ) {
			for ( $i = 0; $i < count( $this->custom_question_answers ); $i ++ ) {
				if ( $this->custom_question_answers[ $i ]->get_id() === $custom_question_answer->get_id() ) {
					$this->custom_question_answers[ $i ] = $custom_question_answer;

					return;
				}
			}

			$this->custom_question_answers[] = $custom_question_answer;
		}
	}

	/**
	 * Gets the sub event responses for this attendee.
	 *
	 * @return array An array of Rsvp_Pro_Attendee_Subevent objects with responses.
	 */
	public function get_sub_event_responses() {
		return $this->sub_event_responses;
	}

	/**
	 * Adds a sub-event response for this attendee.
	 *
	 * @param Rsvp_Pro_Attendee_Subevent $sub_event_response The subevent resposne we want to save.
	 */
	public function add_sub_event_responses( $sub_event_response ) {
		if ( is_a( $sub_event_response, 'Rsvp_Pro_Attendee_Subevent' ) ) {
			for ( $i = 0; $i < count( $this->sub_event_responses ); $i ++ ) {
				if ( ( $this->sub_event_responses[ $i ]->get_id() === $sub_event_response->get_id() ) &&
					 ( $this->sub_event_responses[ $i ]->get_event_id() === $sub_event_response->get_event_id() ) ) {
					$this->sub_event_responses[ $i ] = $sub_event_response;

					return;
				}
			}

			$this->sub_event_responses[] = $sub_event_response;
		}
	}

	/**
	 * Gets the event costs for this attendee.
	 *
	 * @return array An array of Rsvp_Pro_Attendee_Event_costs objects.
	 */
	public function get_event_costs() {
		return $this->event_costs;
	}

	/**
	 * Adds an event cost record for this attendee.
	 *
	 * @param Rsvp_Pro_Attendee_Event_Costs $event_cost The event costs record we want to save.
	 */
	public function add_event_costs( $event_cost ) {
		if ( is_a( $event_cost, 'Rsvp_Pro_Attendee_Event_Costs' ) ) {
			for ( $i = 0; $i < count( $this->event_costs ); $i ++ ) {
				if ( ( $this->event_costs[ $i ]->get_id() === $event_cost->get_id() ) &&
				    ( $this->event_costs[ $i ]->get_rsvp_event_id() === $event_cost->get_rsvp_event_id() ) ) {
					$this->event_costs[ $i ] = $event_cost;

					return;
				}
			}

			$this->event_costs[] = $event_cost;
		}
	}

	/**
	 * The change log for an attendee.
	 *
	 * @return array An array of Rsvp_Pro_Attendee_Changelog objects.
	 */
	public function get_changelog() {
		return $this->changelog;
	}

	/**
	 * Adds a change log object for this attendee.
	 *
	 * @param Rsvp_pro_Attendee_Changelog $changelog A change log object to be added to this attendee.
	 */
	public function add_changelog( $changelog ) {
		if ( is_a( $changelog, 'Rsvp_Pro_Attendee_Changelog' ) ) {
			for ( $i = 0; $i < count( $this->changelog ); $i ++ ) {
				if ( $this->changelog[ $i ]->get_id() === $changelog->get_id() ) {
					$this->changelog[ $i ] = $changelog;

					return;
				}
			}

			$this->changelog[] = $changelog;
		}
	}
}
